import os
import sys
import threading
import subprocess
import re
import tkinter as tk
from tkinter import messagebox, ttk
from yt_dlp import YoutubeDL

# === SETUP DOWNLOAD & FFMPEG PATHS ===
BASE_DIR = getattr(sys, '_MEIPASS', os.path.dirname(os.path.abspath(__file__)))
DOWNLOAD_DIR = os.path.join(BASE_DIR, "Downloads")
os.makedirs(DOWNLOAD_DIR, exist_ok=True)

# Look for bundled FFmpeg in ./ffmpeg/bin
FFMPEG_DIR = os.path.join(BASE_DIR, "ffmpeg", "bin")
FFMPEG_LOCATION = FFMPEG_DIR if os.path.isdir(FFMPEG_DIR) else None

# === GLOBAL STATE ===
processed_urls = set()
last_clipboard = ""

# === FUNCTIONS ===

def open_download_folder():
    """Open the Downloads folder in the OS file explorer."""
    if sys.platform.startswith('win'):
        os.startfile(DOWNLOAD_DIR)
    elif sys.platform == 'darwin':
        subprocess.Popen(['open', DOWNLOAD_DIR])
    else:
        subprocess.Popen(['xdg-open', DOWNLOAD_DIR])


def download_worker(urls):
    """Run downloads in a background thread."""
    download_btn.config(state=tk.DISABLED)
    paste_btn.config(state=tk.DISABLED)
    open_btn.config(state=tk.DISABLED)
    watcher_chk.config(state=tk.DISABLED)
    progress_label.config(text="")
    progress_bar['value'] = 0
    progress_bar['maximum'] = 100

    def progress_hook(d):
        if d['status'] == 'downloading':
            pct = d.get('_percent_str', '').strip().replace('%','')
            try:
                progress_bar['value'] = float(pct)
            except ValueError:
                pass
            progress_label.config(text=f"Downloading: {d.get('filename','')} ({d.get('_percent_str','')})")
        elif d['status'] == 'finished':
            progress_label.config(text="Converting to MP3...")
            progress_bar['value'] = 0
        elif d['status'] == 'error':
            progress_label.config(text="Error during download.")

    ydl_opts = {
        'format': 'bestaudio/best',
        'outtmpl': os.path.join(DOWNLOAD_DIR, '%(title)s.%(ext)s'),
        'postprocessors': [{
            'key': 'FFmpegExtractAudio',
            'preferredcodec': 'mp3',
            'preferredquality': '320',
        }],
        'noplaylist': True,
        'progress_hooks': [progress_hook],
    }
    if FFMPEG_LOCATION:
        ydl_opts['ffmpeg_location'] = FFMPEG_LOCATION

    try:
        with YoutubeDL(ydl_opts) as ydl:
            for url in urls:
                ydl.download([url])
        status_label.config(text=f"✅ Successfully downloaded {len(urls)} file(s).", fg="green")
    except Exception as e:
        status_label.config(text=f"❌ Download failed: {e}", fg="red")
    finally:
        download_btn.config(state=tk.NORMAL)
        paste_btn.config(state=tk.NORMAL)
        open_btn.config(state=tk.NORMAL)
        watcher_chk.config(state=tk.NORMAL)
        progress_label.config(text="")


def start_download():
    raw = url_text.get("1.0", tk.END).strip()
    urls = [line.strip() for line in raw.splitlines() if line.strip()]
    if not urls:
        status_label.config(text="❌ Please enter at least one YouTube URL.", fg="red")
        return
    thread = threading.Thread(target=download_worker, args=(urls,), daemon=True)
    thread.start()


def paste_from_clipboard():
    global last_clipboard
    try:
        clip = app.clipboard_get().strip()
        if clip:
            url_text.insert(tk.END, clip + "\n")
            url_text.focus_set()
            url_text.mark_set("insert", tk.END)
            url_text.see("insert")
            last_clipboard = clip
    except tk.TclError:
        status_label.config(text="❌ Could not access clipboard.", fg="red")


def check_clipboard():
    global last_clipboard
    if watcher_var.get():
        try:
            clip = app.clipboard_get().strip()
            if clip != last_clipboard:
                last_clipboard = clip
                # Simple YouTube URL detection
                if re.search(r"(youtu\.be/|youtube\.com/watch\?)", clip):
                    if clip not in processed_urls:
                        processed_urls.add(clip)
                        status_label.config(text=f"🔍 Detected URL: {clip}", fg="blue")
                        threading.Thread(target=download_worker, args=([clip],), daemon=True).start()
        except tk.TclError:
            pass
    app.after(1000, check_clipboard)

# === GUI SETUP ===
app = tk.Tk()
app.title("YouTube → MP3 Downloader")
app.geometry("540x350")
app.resizable(False, False)

# URL input
tk.Label(app, text="Enter one or more YouTube URLs (one per line):").pack(pady=(10,0))
url_text = tk.Text(app, width=70, height=8)
url_text.pack(pady=(0,10))

# Buttons
btn_frame = tk.Frame(app)
btn_frame.pack()
paste_btn = tk.Button(btn_frame, text="📋 Paste from Clipboard", command=paste_from_clipboard)
paste_btn.grid(row=0, column=0, padx=5)
download_btn = tk.Button(btn_frame, text="⬇️ Download MP3(s)", command=start_download)
download_btn.grid(row=0, column=1, padx=5)
open_btn = tk.Button(btn_frame, text="📂 Open Downloads Folder", command=open_download_folder)
open_btn.grid(row=0, column=2, padx=5)

# Clipboard watcher toggle
watcher_var = tk.BooleanVar(value=False)
watcher_chk = tk.Checkbutton(app, text="Enable Clipboard Watcher", var=watcher_var)
watcher_chk.pack(pady=(20,0))

# Progress bar and labels
progress_label = tk.Label(app, text="", anchor="w")
progress_label.pack(fill="x", padx=10, pady=(0,0))
progress_bar = ttk.Progressbar(app, orient="horizontal", length=550, mode="determinate")
progress_bar.pack(padx=10, pady=(5,10))
status_label = tk.Label(app, text="", fg="green", anchor="w")
status_label.pack(fill="x", padx=10, pady=(0,0))

# Start clipboard watcher loop
app.after(1000, check_clipboard)

app.mainloop()
